;; -*- mode: scheme; coding: utf-8 -*-
;; SPDX-License-Identifier: AGPL-3.0-or-later
;; Loko Scheme - an R6RS Scheme compiler
;; Copyright © 2021 Göran Weinholt

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU Affero General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU Affero General Public License for more details.

;; You should have received a copy of the GNU Affero General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
#!r6rs

;;; Memory maps for physical and virtual address allocation

;; A memory map is a data structure that keeps track of memory
;; addresses. It's used to implement the physical memory map, the
;; kernel virtual memory map, and user space virtual memory maps.
;; Memory maps track relatively large areas, they are not meant to
;; track individual objects.

;; TODO: Use a balanced binary tree

(library (loko runtime mmap)
  (export
    make-mmap mmap?
    mmap-top
    mmap-mark!
    mmap-filter->list
    mmap-for-each

    area?
    area-base area-top area-protection area-type area-info
    area-size)
  (import
    (rnrs (6))
    (rnrs mutable-pairs (6)))

;; Represents the addresses [0,top]. Areas that are missing from the
;; areas field are considered to be unused.
(define-record-type mmap
  (sealed #t)
  (fields top
          (mutable areas)               ;ordered list of areas
          ;; map-callback
          ;; unmap-callback
          )
  (protocol
   (lambda (p)
     (lambda (top)
       (assert (fxpositive? top))
       (p top '())))))

(define-record-type area
  (sealed #t)
  (fields (mutable base)
          (mutable top)
          (mutable protection)
          type
          info))

(define (area-size area)
  (fx+ 1 (fx- (area-top area) (area-base area))))

(define (insert-area-link! mmap prev next)
  (if (not prev)
      (mmap-areas-set! mmap next)
      (set-cdr! prev next)))

(define (mmap-mark! mmap base size protection type info)
  (assert (fx<=? 0 base (fx- (mmap-top mmap) 1)))
  (assert (fx>? size 0))
  (let ((top (fx+ base (fx- size 1))))
    (assert (fx<=? 0 top (mmap-top mmap)))
    (let ((A (make-area base top protection type info)))
      (let lp ((prev #f) (areas (mmap-areas mmap)))
        (cond
          ((null? areas)
           ;; No area overlaps with this one, so insert it at the end.
           (insert-area-link! mmap prev (list A)))

          (else
           (let ((B (car areas)))
             (cond
               ((fx>? (area-base A) (area-top B))
                (lp areas (cdr areas)))

               ((fx<? (area-top A) (area-base B))
                ;; A does not overlap with B, insert it
                (insert-area-link! mmap prev (cons A areas)))

               ((and (fx<=? (area-base A) (area-base B))
                     (fx>=? (area-top A) (area-top B)))
                ;; if A completely overlaps B, remove B
                ;;
                ;;  BBBB   BB     BB     BB
                ;;  AAAA   AAAA  AAAA  AAAA
                (let ((next (cdr areas)))
                  (insert-area-link! mmap prev next)
                  (lp prev next)))

               ((fx<=? (area-base A) (area-base B))
                ;; if A overlaps the start of B, remove
                ;; that part of B and insert A before it
                ;;
                ;;    BBBB     BBBB
                ;;  AAAA       AA
                ;;    =>       =>
                ;;  AAAABB     AABB
                (area-base-set! B (fx+ 1 (area-top A)))
                (insert-area-link! mmap prev (cons A areas)))

               ((and (fx>=? (area-base A) (area-base B))
                     (fx<? (area-top A) (area-top B)))
                ;; if A overlaps the middle of B, remove
                ;; that part of B, splitting it B and C
                ;; BBBBBB
                ;;   AA
                ;;   =>
                ;; BBAACC
                (let ((C (make-area (fx+ 1 (area-top A)) (area-top B)
                                    (area-protection B)
                                    (area-type B)
                                    (area-info B))))
                  (area-top-set! B (fx+ -1 (area-base A)))
                  (insert-area-link! mmap prev (cons B (cons A (cons C (cdr areas)))))))

               (else
                ;; if R overlaps the end of B, remove that
                ;; part of B and insert A after it
                (area-top-set! B (fx+ -1 (area-base A)))
                (insert-area-link! mmap prev (cons B (cons A (cdr areas)))))))))))))

(define (mmap-filter->list proc mmap)
  (filter proc (mmap-areas mmap)))

(define (mmap-for-each proc mmap)
  (for-each proc (mmap-areas mmap))))
