;; -*- mode: scheme; coding: utf-8 -*-
;; SPDX-License-Identifier: EUPL-1.2+
;; This file is a part of Loko Scheme, an R6RS Scheme system
;; Copyright © 2022 Göran Weinholt
#!r6rs

;;; Tests for (loko valand internal)

(import
  (rnrs)
  (loko valand internal))

(define (test-rectangle-intersection)
  (define (make-xorshift32 seed)
    ;; http://www.jstatsoft.org/v08/i14/paper
    (let ((state seed))
      (lambda ()
        (let* ((y state)
               (y (fxxor y (fxarithmetic-shift y 13)))
               (y (fxxor y (fxarithmetic-shift y -17)))
               (y (fxxor y (fxarithmetic-shift y 5)))
               (y (fxand y #xffffffff)))
          (set! state y)
          y))))
  (define random-u32 (make-xorshift32 2463534242))
  (define (rand max)          ;no particular distribution
    (mod (random-u32) max))
  (define w 50)
  (define h 25)
  (define stride w)
  (define disp (make-vector (* h stride) 0))
  (define (print-display mtx w h stride)
    (do ((y 0 (+ y 1)))
        ((= y h))
      (do ((x 0 (+ x 1)))
          ((= x w))
        (display (vector-ref mtx (+ x (* stride y)))))
      (newline)))
  (define (incr-rect! mtx x y w h stride)
    (do ((y y (+ y 1))
         (yend (+ y h)))
        ((= y yend))
      (do ((x x (+ x 1))
           (xend (+ x w)))
          ((= x xend))
        (let ((idx (+ x (* stride y))))
          (vector-set! mtx idx (+ 1 (vector-ref mtx idx)))))))
  (define (rand-rect w h)
    (let* ((x (rand (- w 1)))
           (y (rand (- h 1)))
           (w (+ 1 (rand (- w x 0))))
           (h (+ 1 (rand (- h y 0)))))
      (values x y w h)))
  (define (find-rect mtx w h stride value)
    (let ((rx1 w) (ry1 h) (rx2 0) (ry2 0))
      (do ((y 0 (+ y 1)))
          ((= y h))
        (do ((x 0 (+ x 1)))
            ((= x w))
          (let ((idx (+ x (* stride y))))
            (when (= value (vector-ref mtx idx))
              (set! rx1 (min rx1 x))
              (set! ry1 (min ry1 y))
              (set! rx2 (max rx2 (+ 1 x)))
              (set! ry2 (max ry2 (+ 1 y)))))))
      (if (eqv? rx2 0)
          (values 0 0 0 0)              ;no rect found
          (values rx1 ry1 (- rx2 rx1) (- ry2 ry1)))))
  (do ((i 0 (+ i 1)))
      ((eqv? i 500000))
    (let*-values ([(x1 y1 w1 h1) (rand-rect w h)]
                  [(x2 y2 w2 h2) (rand-rect w h)])
      ;; (write (list (list x1 y1 w1 h1) (list x2 y2 w2 h2)))
      ;; (newline)
      (when #f
        (vector-fill! disp 0)
        (incr-rect! disp x1 y1 w1 h1 stride)
        (display "1:\n")
        (print-display disp w h stride)
        (vector-fill! disp 0)
        (incr-rect! disp x2 y2 w2 h2 stride)
        (display "2:\n")
        (print-display disp w h stride))
      (vector-fill! disp 0)
      (incr-rect! disp x1 y1 w1 h1 stride)
      (incr-rect! disp x2 y2 w2 h2 stride)
      ;; Find the expected intersection and the computed intersection
      (let*-values ([(ex ey ew eh) (find-rect disp w h stride 2)]
                    [(cx cy cw ch) (rectangle-intersection x1 y1 w1 h1 x2 y2 w2 h2)])
        (unless (eqv? cw 0)
          (when #f
            (print-display disp w h stride)
            (newline)))
        (cond ((equal? (list ex ey ew eh)
                       (list cx cy cw ch)))
              (else
               (display "WRONG RESULT!\n")
               (print-display disp w h stride)
               (write `(rectangle-intersection ,x1 ,y1 ,w1 ,h1 ,x2 ,y2 ,w2 ,h2))
               (display " => ")
               (write (list cx cy cw ch))
               (display "\nExpected:\n")
               (write (list ex ey ew eh))
               (newline)
               (error #f "Wrong result")))))))

(test-rectangle-intersection)

(display "OK\n")
