;; -*- mode: scheme; coding: utf-8 -*-
;; SPDX-License-Identifier: EUPL-1.2+
;; Copyright © 2019-2020 G. Weinholt
#!r6rs

;;; Dump ethernet traffic

(import
  (rnrs (6))
  (only (loko) machine-type)
  (loko drivers net)
  (loko drivers net eepro100)
  (loko drivers net rtl8139)
  (loko drivers net rtl8169)
  (loko drivers net tun)
  (loko drivers net virtio)
  (loko drivers pci)
  (loko system logging)
  (loko system unsafe)
  (loko system fibers)

  (text-mode console)
  (text-mode console events)
  (text-mode console model))

(define (fxalign i alignment)
  (fxand (fx+ i (fx- alignment 1))
         (fx- alignment)))

;;; Bare bones VGA text mode output

(define (vga-textmode-backend)
  (define mem-base #xb8000)
  (define reg-base #x3c0)
  (define rows 25)
  (define cols 80)
  ;; VGA regs
  (define crtc-addr 20)
  (define crtc-data 21)
  ;; Registers in the CRT Controller:
  (define cursor-start #x0a)
  (define cursor-end #x0b)
  (define cursor-location-high #x0e)
  (define cursor-location-low #x0f)
  (define (crtc-read addr)
    (put-i/o-u8 (fx+ reg-base crtc-addr) addr)
    (get-i/o-u8 (fx+ reg-base crtc-data)))
  (define (crtc-write addr byte)
    (put-i/o-u8 (fx+ reg-base crtc-addr) addr)
    (put-i/o-u8 (fx+ reg-base crtc-data) byte))
  (define (vga-cursor-move x y)
    (let ((offset (+ x (* y cols))))
      (crtc-write cursor-location-low (fxbit-field offset 0 8))
      (crtc-write cursor-location-high (fxbit-field offset 8 16))))
  (define vga-colors
    (vector Black Blue Green Cyan Red Magenta Brown Gray
            DarkGray LightBlue LightGreen LightCyan LightRed
            LightMagenta Yellow White))
  (define (closest-color col fallback)
    (if (eqv? col Default)
        fallback
        (let lp ((i 0))
          (cond ((fx=? i (vector-length vga-colors)) fallback)
                ((fx=? col (vector-ref vga-colors i)) i)
                (else (lp (fx+ i 1)))))))
  (lambda (cmd arg)
    (case cmd
      [(get-size)
       (values cols rows 0 0)]
      [(init)
       #f]
      [(update redraw)
       (let ((c arg))
         (assert (fx=? cols (console-full-cols c)))
         (assert (fx=? rows (console-full-rows c)))
         (do ((y 0 (fx+ y 1))) ((fx=? y rows))
           (let ((mem-row (fx* 2 (fx* y cols))))
             (when (console-row-dirty? c y 'absolute)
               (let-values ([(buf mbuf fgbuf bgbuf abuf idx) (%index/nowin c 0 y)])
                 (do ((x 0 (fx+ x 1))) ((fx=? x cols))
                   (let ((mem-offset (fx+ (fx* 2 x) mem-row)))
                     (let ((ch (text-ref buf (fx+ idx x))))
                       (unless (textcell-unused? ch)
                         (put-mem-u8 (fx+ mem-base mem-offset)
                                     (if (eqv? ch #\│)
                                         #xB3  ;VGA ROM font...
                                         (char->integer ch)))
                         (let ((fg (closest-color (fg-ref fgbuf (fx+ idx x)) 7))
                               (bg (closest-color (bg-ref bgbuf (fx+ idx x)) 0)))
                           (put-mem-u8 (fx+ mem-base (fx+ mem-offset 1))
                                       (fxior (fxarithmetic-shift-left bg 4)
                                              fg)))))))))))
         (clear-console-dirty! c)
         (vga-cursor-move (fx+ (console-x c) (console-x1 c))
                          (fx+ (console-y c) (console-y1 c))))]
      [(read-event)
       #f]
      [else
       #f])))

(define (draw-ui)
  (reset-window)
  (text-color Default)
  (text-background Default)
  (clrscr)

  (gotoxy 0 0)
  (text-color Yellow)
  (text-background Blue)
  (print "etherdump.sps - Ethernet frame printer demo for Loko Scheme")
  (clreol)

  (gotoxy 0 (- (window-maxy) 1))
  (clreol)

  (text-color Default)
  (text-background Default)
  (set-window 0 1 (window-maxx) (- (window-maxy) 1))
  (clrscr))

(when (eq? (vector-ref (machine-type) 1) 'pc)
  (current-console (make-console (vga-textmode-backend))))
(draw-ui)

(define sizeof-ethhdr 14)

(define (handle-packet buf len)
  (define (hexdump start end)
    (define (bytes start end)
      (text-color LightCyan)
      (do ((i start (fx+ i 1)))
          ((fx=? i (fx+ start 4)))
        (cond ((fx<? i end)
               (let ((b (get-mem-u8 i)))
                 (when (fx<? b #x10) (print "0"))
                 (print (number->string b 16))))
              (else (print "  ")))
        (when (fxodd? i)
          (print #\space)))
      (text-color DarkGray))

    (do ((i start (fx+ i 16)))
        ((fx>=? i end))
      (text-color Default)
      (let ((offset (fx- (fx- i buf) sizeof-ethhdr)))
        (when (fx<? offset #x10) (print "0"))
        (when (fx<? offset #x100) (print "0"))
        (when (fx<? offset #x1000) (print "0"))
        (print (number->string offset 16))
        (print ": "))
      (bytes i end)
      (print "│ ")
      (bytes (fx+ i 4) end)
      (print "│ ")
      (bytes (fx+ i 8) end)
      (print "│ ")
      (bytes (fx+ i 12) end)
      (print "  ")
      (text-color Default)
      (do ((j i (fx+ j 1)))
          ((or (fx=? j (fx+ i 16)) (fx=? j end)))
        (print (integer->char (get-mem-u8 j))))
      (println)))
  (define (print-mac start)
    (do ((i 0 (fx+ i 1)))
        ((fx=? i 6))
      (let ((b (get-mem-u8 (fx+ start i))))
        (when (fx<? b #x10) (print "0"))
        (print (number->string b 16))
        (unless (fx=? i 5)
          (print ":")))))

  (when (fx>? len 12)
    (text-color LightGreen)
    (print-mac (fx+ buf 6))
    (text-color Default)
    (print " -> ")
    (text-color LightGreen)
    (print-mac buf)
    (print " ")
    (text-color Yellow)
    (let ((ethertype (fxior (fxarithmetic-shift-left (get-mem-u8 (fx+ buf 12)) 8)
                            (get-mem-u8 (fx+ buf 13)))))
      (print
       (case ethertype
         ((#x806) 'ARP)
         ((#x800) 'IPv4)
         ((#x86DD) 'IPv6)
         (else
          (number->string ethertype 16))))
      (print " ")
      (text-color LightGreen)
      (print len))
    (println)

    (hexdump (fx+ buf sizeof-ethhdr) (fx+ buf len))))

;; Receive and print packets
(define (receive-loop iface)
  (let ((mac (iface-get-hwaddr iface)))
    (print "Hardware address: ")
    (do ((i 0 (+ i 1)))
        ((= i 6))
      (let ((b (bytevector-u8-ref mac i)))
        (when (< b #x10)
          (print "0"))
        (print (number->string b 16))
        (unless (= i 5)
          (print ":"))))
    (println))
  (let lp ()
    (let ((pkt (get-message (iface-rx-ch iface))))
      (let ((buf0 (car (netpkt-bufs pkt))))
        (handle-packet (car buf0) (cdr buf0)))
      (netpkt-free! pkt)
      (lp))))

(case (vector-ref (machine-type) 1)
  ((pc)
   (let ((*devs* (pci-scan-bus #f)))
     (println "Scanning the PCI bus for supported network cards...")
     (update-screen)

     (let ((devs *devs*))
       (or
         (find
          (lambda (dev)
            (cond ((probe·pci·rtl8169? dev)
                   (print "Found an rtl8169 on ")
                   (println (list (pcidev-bus dev) (pcidev-dev dev) (pcidev-func dev)))
                   (let ((iface (make-iface 'ethernet)))
                     (spawn-fiber (lambda ()
                                    (receive-loop iface)))
                     (spawn-fiber (lambda ()
                                    (driver·pci·rtl8169 dev iface))))
                   #t)
                  ((probe·pci·rtl8139? dev)
                   (print "Found an rtl8139 on ")
                   (println (list (pcidev-bus dev) (pcidev-dev dev) (pcidev-func dev)))
                   (let ((iface (make-iface 'ethernet)))
                     (spawn-fiber (lambda ()
                                    (receive-loop iface)))
                     (spawn-fiber (lambda ()
                                    (driver·pci·rtl8139 dev iface))))
                   #t)
                  ((probe·pci·eepro100? dev)
                   (print "Found an eepro100 on ")
                   (println (list (pcidev-bus dev) (pcidev-dev dev) (pcidev-func dev)))
                   (let ((iface (make-iface 'ethernet)))
                     (spawn-fiber (lambda ()
                                    (receive-loop iface)))
                     (spawn-fiber (lambda ()
                                    (driver·pci·eepro100 dev iface))))
                   #t)
                  ((probe·pci·virtio-net? dev)
                   (print "Found a virtio network card on ")
                   (println (list (pcidev-bus dev) (pcidev-dev dev) (pcidev-func dev)))
                   (let ((iface (make-iface 'ethernet)))
                     (spawn-fiber (lambda ()
                                    (receive-loop iface)))
                     (spawn-fiber (lambda ()
                                    (driver·pci·virtio-net dev iface))))
                   #t)
                  (else #f)))
          devs)
         (println "No network card found!")))))
  (else
   (let ((iface (make-iface 'ethernet))
         (devname (if (null? (cdr (command-line)))
                      "tap0"
                      (cadr (command-line)))))
     (print "Opening the TAP interface ")
     (println devname)
     (update-screen)
     (spawn-fiber (lambda ()
                    (receive-loop iface)))
     (spawn-fiber (lambda ()
                    (driver·kernel·tun devname iface))))))

(current-log-callback
 (lambda (e)
   (print "[")
   (print (cdr (assq 'SEVERITY e)))
   (print "] ")
   (println (cdr (assq 'MESSAGE e)))))

;; Keep the screen updated. Not multiprocessing safe, but fibers are
;; cooperative in Loko.
(let lp ()
  (sleep 0.1)
  (update-screen)
  (lp))
