#!/usr/bin/env scheme-script
;; -*- mode: scheme; coding: utf-8 -*- !#
;; SPDX-License-Identifier: EUPL-1.2+
;; This file is a part of Loko Scheme, an R6RS Scheme system
;; Copyright © 2023 G. Weinholt
#!r6rs

;;; Test suite that verifies that eof behavior on ports

;; Among other things.

(import
  (rnrs)
  (rnrs mutable-pairs)
  (rnrs mutable-strings))

(define (subbytevector bv start end)
  (let ((ret (make-bytevector (- end start))))
    (bytevector-copy! bv start ret 0 (- end start))
    ret))

(define (string-copy! s ss t ts k)
  (assert (not (eq? t s)))
  (do ((i (fx- k 1) (fx- i 1))
       (ti ts (fx+ ti 1))
       (si ss (fx+ si 1)))
      ((eqv? i -1))
    (string-set! t ti (string-ref s si))))

(define (make-weird-port id x*)
  (define (read! bv start count)
    (assert (not (zero? count)))
    (cond ((null? x*)
           0)
          (else
           (let ((x (car x*)))
             ;; (display "[Port OP: ")
             ;; (write x)
             ;; (display "]")
             (cond
               ((bytevector? x)
                (let ((n (min count (bytevector-length x))))
                  (bytevector-copy! x 0 bv start n)
                  (cond ((= (bytevector-length x) n)
                         (set! x* (cdr x*)))
                        (else
                         (set-car! x* (subbytevector x n (bytevector-length x)))))
                  n))
               (else                    ;byte
                (set! x* (cdr x*))
                (cond ((not x)
                       0)
                      (else
                       (bytevector-u8-set! bv start x)
                       1))))))))
  (make-custom-binary-input-port id read! #f #f #f))

(define (make-weird-textual-port id x*)
  (define (read! bv start count)
    (assert (not (zero? count)))
    (cond ((null? x*)
           0)
          (else
           (let ((x (car x*)))
             ;; (display "[Port OP: ")
             ;; (write x)
             ;; (display "]")
             (cond
               ((string? x)
                (let ((n (min count (string-length x))))
                  (string-copy! x 0 bv start n)
                  (cond ((= (string-length x) n)
                         (set! x* (cdr x*)))
                        (else
                         (set-car! x* (substring x n (string-length x)))))
                  n))
               (else                    ;char
                (set! x* (cdr x*))
                (cond ((not x)
                       0)
                      (else
                       (string-set! bv start x)
                       1))))))))
  (make-custom-textual-input-port id read! #f #f #f))

(define ERRORS 0)

(define (cmp found* expect*)
  (display "Expect: ")
  (write expect*)
  (newline)
  (display "Found:  ")
  (write (map (lambda (x) (if (eof-object? x) 'eof x)) found*))
  (newline)
  (for-each (lambda (found expect)
              (cond ((eq? expect 'eof)
                     (unless (eof-object? found)
                       (display "ERROR! Expected the end-of-file but found ")
                       (write found)
                       (newline)
                       (set! ERRORS (+ ERRORS 1))))
                    (else
                     (unless (equal? found expect)
                       (display "Expected ")
                       (write expect)
                       (display " but found ")
                       (write found)
                       (newline)
                       (set! ERRORS (+ ERRORS 1))))))
            found* expect*))

(display "test get-bytevector-n\n")
(let* ((p (make-weird-port "n" '(0 1 #f 2 3 #f 4 5)))
       (x0 (get-bytevector-n p 4))
       (x1 (get-bytevector-n p 4))
       (x2 (get-bytevector-n p 4))
       (x3 (get-bytevector-n p 4))
       (x4 (get-bytevector-n p 4))
       (x5 (get-bytevector-n p 4))
       (x6 (get-bytevector-n p 4)))
  (write (list x0 x1 x2 x3 x4 x5 x6))
  (newline)
  (cmp (list x0 x1 x2 x3 x4 x5 x6)
       '(#vu8(0 1) eof #vu8(2 3) eof #vu8(4 5) eof eof)))
(newline)

(display "test get-string-n\n")
(let* ((p (make-weird-textual-port "n" '(#\a #\b #f #\c #\d #f #\e #\f)))
       (x0 (get-string-n p 4))
       (x1 (get-string-n p 4))
       (x2 (get-string-n p 4))
       (x3 (get-string-n p 4))
       (x4 (get-string-n p 4))
       (x5 (get-string-n p 4))
       (x6 (get-string-n p 4)))
  (write (list x0 x1 x2 x3 x4 x5 x6))
  (newline)
  (cmp (list x0 x1 x2 x3 x4 x5 x6)
       '("ab" eof "cd" eof "ef" eof eof)))
(newline)

(display "test get-bytevector-all\n")
(let* ((p (make-weird-port "all" '(0 1 #f 2 3 #f 4 5)))
       (x0 (get-bytevector-all p))
       (x1 (get-bytevector-all p))
       (x2 (get-bytevector-all p))
       (x3 (get-bytevector-all p))
       (x4 (get-bytevector-all p)))
  (cmp (list x0 x1 x2 x3 x4)
       '(#vu8(0 1) #vu8(2 3) #vu8(4 5) eof eof)))
(newline)

(display "test get-string-all\n")
(let* ((p (make-weird-textual-port "all" '(#\a #\b #f #\c #\d #f #\e #\f)))
       (x0 (get-string-all p))
       (x1 (get-string-all p))
       (x2 (get-string-all p))
       (x3 (get-string-all p))
       (x4 (get-string-all p)))
  (cmp (list x0 x1 x2 x3 x4)
       '("ab" "cd" "ef" eof eof)))
(newline)

(display "test get-line\n")
(let* ((p (make-weird-textual-port "line" '(#\a #\b #f #\c #\d #f #\e #\f #\newline)))
       (x0 (get-line p))
       (x1 (get-line p))
       (x2 (get-line p))
       (x3 (get-line p))
       (x4 (get-line p))
       (x5 (get-line p))
       (x6 (get-line p)))
  (cmp (list x0 x1 x2 x3 x4 x5 x6)
       '("ab" eof "cd" eof "ef" eof eof)))
(newline)

(display "test get-line 2\n")
(let* ((p (make-weird-textual-port "line" '(#\a #\b #\newline
                                            #\c #\d #\newline
                                            #\newline)))
       (x0 (get-line p))
       (x1 (get-line p))
       (x2 (get-line p))
       (x3 (get-line p))
       (x4 (get-line p)))
  (cmp (list x0 x1 x2 x3 x4)
       '("ab" "cd" "" eof eof)))
(newline)

(display "test get-bytevector-n!\n")
(let* ((p (make-weird-port "n!" '(0 1 #f 2 3 #f 4 5)))
       (b0 (make-bytevector 4 0)) (n0 (get-bytevector-n! p b0 0 4))
       (b1 (make-bytevector 4 0)) (n1 (get-bytevector-n! p b1 0 4))
       (b2 (make-bytevector 4 0)) (n2 (get-bytevector-n! p b2 0 4))
       (b3 (make-bytevector 4 0)) (n3 (get-bytevector-n! p b3 0 4))
       (b4 (make-bytevector 4 0)) (n4 (get-bytevector-n! p b4 0 4))
       (b5 (make-bytevector 4 0)) (n5 (get-bytevector-n! p b5 0 4))
       (b6 (make-bytevector 4 0)) (n6 (get-bytevector-n! p b6 0 4)))
  (cmp (list n0 n1 n2 n3 n4 n5 n6)
       '(2 eof 2 eof 2 eof eof))
  (cmp (list b0 b1 b2 b3 b4 b5 b6)
       '(#vu8(0 1 0 0) #vu8(0 0 0 0) #vu8(2 3 0 0) #vu8(0 0 0 0) #vu8(4 5 0 0) #vu8(0 0 0 0) #vu8(0 0 0 0))))
(newline)

(display "test get-string-n!\n")
(let* ((p (make-weird-textual-port "n!" '(#\a #\b #f #\c #\d #f #\e #\f)))
       (b0 (make-string 4 #\-)) (n0 (get-string-n! p b0 0 4))
       (b1 (make-string 4 #\-)) (n1 (get-string-n! p b1 0 4))
       (b2 (make-string 4 #\-)) (n2 (get-string-n! p b2 0 4))
       (b3 (make-string 4 #\-)) (n3 (get-string-n! p b3 0 4))
       (b4 (make-string 4 #\-)) (n4 (get-string-n! p b4 0 4))
       (b5 (make-string 4 #\-)) (n5 (get-string-n! p b5 0 4))
       (b6 (make-string 4 #\-)) (n6 (get-string-n! p b6 0 4)))
  (cmp (list n0 n1 n2 n3 n4 n5 n6)
       '(2 eof 2 eof 2 eof eof))
  (cmp (list b0 b1 b2 b3 b4 b5 b6)
       '("ab--" "----" "cd--" "----" "ef--" "----" "----")))
(newline)

(display "test get-bytevector-some\n")
(let* ((p (make-weird-port "some" '(0 1 #f 2 3 #f 4 5)))
       (x0 (get-bytevector-some p))
       (x1 (get-bytevector-some p))
       (x2 (get-bytevector-some p))
       (x3 (get-bytevector-some p))
       (x4 (get-bytevector-some p))
       (x5 (get-bytevector-some p))
       (x6 (get-bytevector-some p))
       (x7 (get-bytevector-some p))
       (x8 (get-bytevector-some p))
       (x9 (get-bytevector-some p)))
  (cmp (list x0 x1 x2 x3 x4 x5 x6 x7 x8 x9)
       '(#vu8(0) #vu8(1) eof #vu8(2) #vu8(3) eof #vu8(4) #vu8(5) eof eof)))
(newline)

(display "test get-u8\n")
(let* ((p (make-weird-port "u8" '(0 1 #f 2 3 #f #f 4 5)))
       (x0 (get-u8 p))
       (x1 (get-u8 p))
       (x2 (get-u8 p))
       (x3 (get-u8 p))
       (x4 (get-u8 p))
       (x5 (get-u8 p))
       (x6 (get-u8 p))
       (x7 (get-u8 p))
       (x8 (get-u8 p))
       (x9 (get-u8 p)))
  (cmp (list x0 x1 x2 x3 x4 x5 x6 x7 x8 x9)
       '(0 1 eof 2 3 eof eof 4 5 eof)))
(newline)

(display "test get-char\n")
(let* ((p (make-weird-textual-port "char" '(#\a #\b #f #\c #\d #f #\e #\f)))
       (x0 (get-char p))
       (x1 (get-char p))
       (x2 (get-char p))
       (x3 (get-char p))
       (x4 (get-char p))
       (x5 (get-char p))
       (x6 (get-char p))
       (x7 (get-char p))
       (x8 (get-char p))
       (x9 (get-char p)))
  (cmp (list x0 x1 x2 x3 x4 x5 x6 x7 x8 x9)
       '(#\a #\b eof #\c #\d eof #\e #\f eof eof)))
(newline)

(display "test lookahead-u8\n")
(define (lookahead-u8* p)
  (let ((x (lookahead-u8 p)))
    (get-u8 p)
    x))
(let* ((p (make-weird-port "lu8" '(0 1 #f 2 #f 4)))
       (x0 (lookahead-u8  p))
       (x1 (lookahead-u8* p))
       (x2 (lookahead-u8  p))
       (x3 (lookahead-u8* p))
       (x4 (lookahead-u8  p))
       (x5 (lookahead-u8* p))
       (x6 (lookahead-u8  p))
       (x7 (lookahead-u8* p))
       (x8 (lookahead-u8  p))
       (x9 (lookahead-u8* p))
       (x10 (lookahead-u8  p))
       (x11 (lookahead-u8* p))
       (x12 (lookahead-u8  p))
       (x13 (lookahead-u8* p)))
  (cmp (list x0 x1 x2 x3 x4 x5 x6 x7 x8 x9 x10 x11 x12 x13)
       '(0 0 1 1 eof eof 2 2 eof eof 4 4 eof eof)))
(newline)

(display "test lookahead-char\n")
(define (lookahead-char* p)
  (let ((x (lookahead-char p)))
    (get-char p)
    x))
(let* ((p (make-weird-textual-port "lchar" '(#\a #\b #f #\c #f #\d)))
       (x0 (lookahead-char  p))
       (x1 (lookahead-char* p))
       (x2 (lookahead-char  p))
       (x3 (lookahead-char* p))
       (x4 (lookahead-char  p))
       (x5 (lookahead-char* p))
       (x6 (lookahead-char  p))
       (x7 (lookahead-char* p))
       (x8 (lookahead-char  p))
       (x9 (lookahead-char* p))
       (x10 (lookahead-char p))
       (x11 (lookahead-char* p))
       (x12 (lookahead-char  p))
       (x13 (lookahead-char* p)))
  (cmp (list x0 x1 x2 x3 x4 x5 x6 x7 x8 x9 x10 x11 x12 x13)
       '(#\a #\a #\b #\b eof eof #\c #\c eof eof #\d #\d eof eof)))
(newline)

(display "test 4096 read\n")
(let* ((p (make-weird-port "read-4k" (list (make-bytevector 4096 47) #f 1)))
       (x0 (get-bytevector-n p 4096))
       (x1 (get-bytevector-n p 4096))
       (x2 (get-bytevector-n p 4096))
       (x3 (get-bytevector-n p 4096)))
  (cmp (list x0 x1 x2 x3)
       (list (make-bytevector 4096 47) 'eof #vu8(1) 'eof)))
(newline)

(display "test 1+4096 read\n")
(let* ((p (make-weird-port "read-1+4k" (list (make-bytevector 4096 47) #f 1)))
       (x0 (get-u8 p))
       (x1 (get-bytevector-n p 4096))
       (x2 (get-bytevector-n p 4096))
       (x3 (get-bytevector-n p 4096))
       (x4 (get-bytevector-n p 4096)))
  (cmp (list x0 x1 x2 x3 x4)
       (list 47 (make-bytevector 4095 47) 'eof #vu8(1) 'eof)))
(newline)


(define (make-xorshift32 seed)
  ;; http://www.jstatsoft.org/v08/i14/paper
  (let ((state seed))
    (lambda ()
      (let* ((y state)
             (y (fxxor y (fxarithmetic-shift y 13)))
             (y (fxxor y (fxarithmetic-shift y -17)))
             (y (fxxor y (fxarithmetic-shift y 5)))
             (y (fxand y #xffffffff)))
        (set! state y)
        y))))

(define (make-binary-port-sequence len rand32)
  (do ((i 0 (+ i 1))
       (seq '() (cons (let ((x (mod (rand32) 15)))
                        (case x
                          ((0) #f)
                          ((1)
                           (let ((len (mod (rand32) 32)))
                             (do ((bv (make-bytevector len))
                                  (i 0 (+ i 1)))
                                 ((= i (bytevector-length bv))
                                  bv)
                               (bytevector-u8-set! bv i (bitwise-and (rand32) #xff)))))
                          (else x)))
                      seq)))
      ((= i len) seq)))

(define (make-binary-i/o-sequence len rand32)
  (do ((i 0 (+ i 1))
       (ops '() (cons (let ((x (mod (rand32) 5)))
                        (case x
                          ((0)
                           (let ((n (mod (rand32) 31)))
                             (cons (letrec ((get-bytevector-n-op
                                             (lambda (p) (get-bytevector-n p n))))
                                     get-bytevector-n-op)
                                   `(get-bytevector-n p ,n))))
                          ((1)
                           (cons (letrec ((lookahead-u8-op
                                           (lambda (p) (lookahead-u8 p))))
                                   lookahead-u8-op)
                                 `(lookahead-u8 p)))
                          ((2)
                           (let ((n (mod (rand32) 31)))
                             (cons (letrec ((get-bytevector-n!-op
                                             (lambda (p)
                                               (let ((bv (make-bytevector n 47)))
                                                 (cons (get-bytevector-n! p bv 0 n)
                                                       bv)))))
                                     get-bytevector-n!-op)
                                   `(get-bytevector-n! p 0 ,n))))
                          ((3) (cons port-eof? `(port-eof? p)))
                          ((4)
                           (cons (letrec ((get-bytevector-some-op
                                           (lambda (p)
                                             (let ((bv (get-bytevector-some p)))
                                               (cons (and (bytevector? bv)
                                                          (bytevector-length bv))
                                                     bv)))))
                                   get-bytevector-some-op)
                                 `(get-bytevector-some p)))
                          (else (cons (letrec ((get-u8-op
                                                (lambda (p) (get-u8 p))))
                                        get-u8-op)
                                      `(get-u8 p)))))
                      ops)))
      ((= i len) ops)))

(define (make-textual-port-sequence len rand32)
  (do ((i 0 (+ i 1))
       (seq '() (cons (let ((x (mod (rand32) 7)))
                        (case x
                          ((0) #f)
                          ((1)
                           (let ((len (mod (rand32) 32)))
                             (make-string len
                                          (integer->char (+ (char->integer #\a)
                                                            (mod (rand32) 27))))))
                          (else
                           (integer->char (+ (char->integer #\a) x)))))
                      seq)))
      ((= i len) seq)))

(define (make-textual-i/o-sequence len rand32)
  (do ((i 0 (+ i 1))
       (ops '() (cons (let ((x (mod (rand32) 5)))
                        (case x
                          ((0)
                           (let ((n (mod (rand32) 31)))
                             (cons (letrec ((get-string-n-op
                                             (lambda (p) (get-string-n p n))))
                                     get-string-n-op)
                                   `(get-string-n p ,n))))
                          ((1)
                           (cons (letrec ((lookahead-char-op
                                           (lambda (p) (lookahead-char p))))
                                   lookahead-char-op)
                                 `(lookahead-char p)))
                          ((2)
                           (let ((n (mod (rand32) 31)))
                             (cons (letrec ((get-string-n!-op
                                             (lambda (p)
                                               (let ((str (make-string n #\X)))
                                                 (cons (get-string-n! p str 0 n)
                                                       str)))))
                                     get-string-n!-op)
                                   `(get-string-n! p 0 ,n))))
                          ((3) (cons port-eof? `(port-eof? p)))
                          (else (cons (letrec ((get-char-op
                                                (lambda (p) (get-char p))))
                                        get-char-op)
                                      `(get-char p)))))
                      ops)))
      ((= i len) ops)))

(display "Generating random binary operations\n")
(let ()
  (define rand32 (make-xorshift32 2463534242))
  (let* ((seq (make-binary-port-sequence 200000 rand32))
         (port (make-weird-port "random" seq))
         (i/o* (make-binary-i/o-sequence 2000 rand32)))
    (for-each (lambda (proc op)
                (display "I/O OP: ")
                (write op)
                (display " => ")
                (let ((x (proc port)))
                  (write x)
                  (newline)
                  x))
              (map car i/o*)
              (map cdr i/o*))))
(newline)

(display "Generating random textual operations\n")
(let ()
  (define rand32 (make-xorshift32 2463534242))
  (let* ((seq (make-textual-port-sequence 200000 rand32))
         (port (make-weird-textual-port "random" seq))
         (i/o* (make-textual-i/o-sequence 2000 rand32)))
    (for-each (lambda (proc op)
                (display "I/O OP: ")
                (write op)
                (display " => ")
                (let ((x (proc port)))
                  (write x)
                  (newline)
                  x))
              (map car i/o*)
              (map cdr i/o*))))
(newline)

(cond ((eqv? ERRORS 0)
       (display "Tests passed\n"))
      (else
       (display "Failures: ")
       (display ERRORS)
       (newline)
       (exit 1)))
