#!/usr/bin/env scheme-script
;; -*- coding: utf-8; mode: scheme -*-
;; Copyright © 2006-2007, 2020 Gwen Weinholt
;; SPDX-License-Identifier: MIT

;;; Convert a .bdf font file into a two-level table

;; Current limitations: only unicode, no bounding box, only 8-bit
;; fonts, nothing special.

;; XXX: PCF or another binary format should be prefered when
;; dynamically loading from a file system.

(import
  (rnrs)
  (wak fmt))

(define (include-codepoint? codepoint)
  #t)

(define (make-bdf-reader port)
  (define (parse-bdf-line line)
    (let ((p (open-string-input-port line)))
      (let ((keyword (read p)))
        (case keyword
          ((COMMENT STARTFONT FONT)
           (get-char p)
           (list keyword (get-line p)))
          (else
           (let lp ((val keyword))
             (if (eof-object? val)
                 '()
                 (cons val (lp (read p))))))))))
  (lambda ()
    (let ((line (get-line port)))
      (if (eof-object? line)
          line
          (parse-bdf-line line)))))

(define (make-bdf-parser p reader)
  (define (read-character)
    (let ((line (reader)))
      (if (not (eq? 'ENCODING (car line)))
          (read-character)
          (let ((encoding (cadr line)))
            (let skip ((line (reader)))
              (if (not (eq? 'BITMAP (car line)))
                  (skip (reader))
                  (cons encoding
                        (u8-list->bytevector
                         (let lp ()
                           (let ((line (get-line p)))
                             (if (equal? line "ENDCHAR")
                                 '()
                                 (cons (string->number line 16) (lp)))))))))))))
  (lambda ()
    (let lp ((line (reader)))
      (case (car line)
        ((STARTCHAR)
         (read-character))
        ((ENDFONT)
         #f)
        ((COMMENT COPYRIGHT FONT)
         ;; Who wins the lazy bastard challenge?
         (fmt #t ";; " (car line) " " (cadr line) nl)
         (lp (reader)))
        (else
         (lp (reader)))))))

(define (bdf->two-stage-table filename)
  (call-with-input-file filename
    (lambda (p)
      (let ((font (make-eqv-hashtable))
            (parser (make-bdf-parser p (make-bdf-reader p))))
        (do ((character (parser) (parser)))
            ((not character))
          (when (include-codepoint? (car character))
            (let ((tablenum (div (car character) #x100)))
              ;; Create the table if it doesn't exist
              (when (not (hashtable-contains? font tablenum))
                (hashtable-set! font tablenum (make-vector 256 #f)))
              (let ((table (hashtable-ref font tablenum #f)))
                (vector-set! table
                             (mod (car character) #x100)
                             (cdr character))))))
        (let ((ret (make-vector (+ 1 (apply max (vector->list (hashtable-keys font))))
                                #f)))
          (let-values ([(keys values) (hashtable-entries font)])
            (vector-for-each (lambda (key value)
                               (vector-set! ret key value))
                             keys values))
          ret)))))

(let ((filename (cadr (command-line)))
      (name (caddr (command-line))))
  (fmt #t
       ";; Two-level font table" nl
       ";; Automatically generated by bdf2sls.sps from " filename nl
       "#!r6rs" nl
       "(library " name nl
       "  (export font)" nl
       "  (import (only (rnrs) define quote))"
       nl)
  (fmt #t (pretty
           `(define font
              ',(bdf->two-stage-table filename))))
  (fmt #t ")" nl))
