# Interrupt handling in Loko Scheme

## In brief

Loko in essence treats device drivers as communicating sequential
processes. IRQs from devices are treated as messages sent from the
device to the driver. Requests to configure the device or transfer
data between the device and the rest of the system is also done with
messages. Device drivers are not fundamentally different from other
Scheme programs.

## Historical background

This document uses the term *interrupt* to denote an interruption in
the processor's normal execution sequence; *IRQ* to denote an
interrupt from a hardware device and *trap* to denote an interrupt
from the processor itself. (There are more types of interrupts, but
they are currently not discussed here).

IRQs are handled differently from how they are handled in normal
kernels. An anecdote from [_The Rise of Worse is Better_][rise] by
Richard P. Gabriel is relevant here:

> Two famous people, one from MIT and another from Berkeley (but
> working on Unix) once met to discuss operating system issues. The
> person from MIT was knowledgeable about ITS (the MIT AI Lab
> operating system) and had been reading the Unix sources. He was
> interested in how Unix solved the PC loser-ing problem. The PC
> loser-ing problem occurs when a user program invokes a system
> routine to perform a lengthy operation that might have significant
> state, such as IO buffers. If an interrupt occurs during the
> operation, the state of the user program must be saved. Because the
> invocation of the system routine is usually a single instruction,
> the PC of the user program does not adequately capture the state of
> the process. The system routine must either back out or press
> forward. The right thing is to back out and restore the user program
> PC to the instruction that invoked the system routine so that
> resumption of the user program after the interrupt, for example,
> re-enters the system routine. It is called PC loser-ing because the
> PC is being coerced into loser mode, where loser is the affectionate
> name for user at MIT.

> The MIT guy did not see any code that handled this case and asked
> the New Jersey guy how the problem was handled. The New Jersey guy
> said that the Unix folks were aware of the problem, but the solution
> was for the system routine to always finish, but sometimes an error
> code would be returned that signaled that the system routine had
> failed to complete its action. A correct user program, then, had to
> check the error code to determine whether to simply try the system
> routine again. The MIT guy did not like this solution because it was
> not the right thing.

 [rise]: https://www.dreamsongs.com/RiseOfWorseIsBetter.html

From the New Jersey guy we get the `errno` value `-EINTR`, so he was
clearly the more successful one. The MIT guy's approach would have
been to carefully design syscalls so that they keep their full state
in a structure or in the arguments to the syscall, which is kind of
like doing a very manual and tedious `call/cc` when an interrupt
arrives. And nobody has time for that.

But when the New Jersey guy feels like even `-EINTR` is too difficult
to manage, we get *uninterruptible sleep*. This is a situation where
programs aren't running, but they can't be killed either. This happens
when a program is blocked in a syscall and is holding an unknown
number of locks and other state in the kernel. Maybe it was reading
from a file, but something got wedged and stopped responding. Instead
of an error code, the process is in uninterruptible sleep. This slowly
creeps from program to program as other parties try to communicate
with the frozen process.

So Loko takes a different approach to all this.

## An experimental approach to IRQs

IRQs are generated by hardware devices when they want the attention of
the processor. An example is a UART (serial port controller) that has
just received a byte. It will generate an IRQ to get the driver to
read the byte from its buffer.

The normal idea of how to handle an IRQ is basically as follows:
install the interrupt service routine (ISR) that comes as part of the
driver, reset the device to its normal state, then enable the IRQ in
the interrupt controller. An ISR is a special piece of code that must
be prepared to run at potentially any time (except when interrupts are
disabled). Usually there is a priority order on IRQs so that they can
in turn be interrupted by higher-priority IRQs. Either way, when an
IRQ arrives the driver quickly services the hardware. In the UART
example it would read a byte from the UART and place it in a software
controlled buffer.

In this way of doing things, there are unfortunately severe
restrictions on what can be done in an ISR. An ISR runs in "interrupt
context" where many usual kernel services are simply unavailable.
Anything that would block the program is usually unavailable and
access to memory is limited. Arranging things so that arbitrary Scheme
code can run in interrupt context is difficult.

For this reason, Loko does not run driver code in ISRs. The ISRs are
instead minimal pieces of code that cooperate with the process
scheduler to make the driver's process runnable. An IRQ is sent as a
message to the process and it handles it at its leisure.

This approach is somewhat experimental, and may have some problems
with latency in legacy hardware such as UARTs, but modern devices do
bus mastering DMA and are generally not sensitive to interrupt
servicing latency. Bus mastering DMA means that the device has access
to the system's memory. Generally such a device cooperates with the
driver to maintain queues in system memory that describes data
transfers.

There are pros and cons to Loko's approach. The cons are that IRQs are
handled with some latency, but this is usually not a problem for
modern devices. The pros are that it makes driver code *much* easier
to write and maintain. Since it eliminates many of the usual
difficulties with writing drivers, it may even mean that most
competent Schemers with access to the hardware programming manual can
write device drivers. (Some difficulties still remain with manual
memory management, but they are not that hard to deal with).

Even if latency is a potential problem, Loko's approach should be good
for throughput. A driver can easily decide that data is coming in at
such a high rate that it doesn't need to use interrupts, and switch
over to periodic polling instead. Linux uses this technique in its
networking stack under the cryptic name "New API".

Another benefit of Loko's approach is that the dilemma in the "worse
is better" story is resolved. User programs never need to be given an
equivalent of `-EINTR` and the programmer does not need to manually
keep track of where they are in the handling of a system call.

## Loko's use of traps

Loko offloads as much error checking as possible on built-in
mechanisms in the hardware. Instead of using explicit type checks, it
lets the hardware do the type checks (where possible).

Programming errors like `(/ 1 0)` are often signalled by the hardware
in most programming environments. Loko takes this further and extends
it to errors like `(car #f)`, `(vector-ref "foo" 0)` and `(1 + 2)`.
Loko uses the processor's alignment checking feature to trap wrong
uses of pairs, procedures, strings, vectors and bytevectors. You can
read more about this in [Faster Dynamic Type Checks][alignment-check].

  [alignment-check]: https://weinholt.se/articles/alignment-check/

## Interrupts on bare hardware AMD64

The interrupt handlers are in `(loko arch amd64 pc-interrupts)` and
are written in assembly.

There are two fundamentally different types of interrupts that both go
under the name interrupt and that use similar mechanisms, but have
very different sources.

### Traps

Traps are interrupts that are triggered by some classes of errors in
the running program. For these interrupts the interrupt handlers cause
the program to invoke an error handler written in Scheme. No care is
taken to preserve the program's current stack frame or current
register values, which means that some useful debugging information is
lost. While that is unfortunate, and should be fixed, it is still
semantically correct since these errors are all categorized as
`&serious`.

Traps are handled by entries in the interrupt descriptor table (IDT).
The IDT controls whether the processor should change privilege levels,
which address it should jump to, which stack it should use and whether
interrupts should be automatically disabled or not.

The processor pushes some of the program state on the stack and gives
control to the handler. For traps, the handlers identify the cause of
the trap and decide which library function should take control over
the program. It then executes a tail-call to that function. These
functions live in `(loko arch amd64 lib-traps)` and are responsible
for calling the error handler, which is written in Scheme.

### IRQs

The IRQ handling in Loko relies on the processor's *interrupt stack
table* (IST) and the interrupt controller's *specific end of
interrupt* (SEOI). On AMD64 systems the SEOI feature is available in
the legacy PIC and the APIC (except in early versions). Interrupt
priorities are not meant to be used.

Interrupt masking is a very important part of Loko's IRQ handling.
There are three places where interrupts can be masked: the device
itself, the interrupt controller and the processor. The driver
configures the device to generate interrupts in a way that suites the
driver. The interrupt controller is responsible for delivering
interrupts to the processor and can be asked by the processor to mask
an interrupt. The interrupt controller also keeps track of which
interrupts are being serviced and temporarily masks them until they
are acknowledged. Finally, the processor can also mask interrupts with
the `IF` bit in the flags register. When `IF` is clear, no interrupts
are delivered.

Loko's IRQ handling system relies on masking interrupts with the `IF`
bit and delaying acknowledgement. `IF` is used to mask interrupts
while the scheduler is running. When inside the scheduler, interrupts
can only happen in one very controlled circumstance: in the `sys_hlt`
syscall. This is used when no processes are runnable and it lets the
processor save energy. The `sys_hlt` syscall returns when an IRQ has
been delivered to the processor. The scheduler then finds the driver
process that has the IRQ registered, enqueues it as a message and
makes the process runnable.

Normal processes always run with interrupts unmasked, except maybe for
some brief and tricky moments during task switching. When an interrupt
arrives, the processor uses the IST to switch to a different stack.
This is necessary to avoid messing up the process's Scheme stack. All
registers are saved in the process control block so that the process
can be resumed later. The IRQ handler resumes the scheduler and lets
it know what happened.

When the process resumes it will have an IRQ number in its message
queue. It is up to the process when it wants to dequeue this message,
but usually a driver process will be waiting for an IRQ to arrive.
Whenever a process calls the scheduler it can also receive a pending
message. The two cases are then that a process either becomes runnable
due to an IRQ and that a process is already runnable and will see the
IRQ later.

Processes that are notified about IRQs will handle them, doing
whatever task is needed to service the IRQ in the hardware, and
afterwards tell the scheduler to acknowledge the IRQ. The scheduler
then sends an instruction to the interrupt controller to acknowledge
that specific interrupt. At this point the device may again choose to
generate an interrupt when the conditions are right.

There are a few classes of programming errors when it comes to IRQ
code in drivers. The driver may decide that it has handled an IRQ and
sends an acknowledgement. However, if it missed an interrupt reason,
the device may generate the same IRQ again, immediately after
acknowledgement. This slows down the system with unnecessary work in
the driver.

The opposite can also happen if a driver does not properly handle all
the interrupt reasons. The symptom can be that an IRQ arrives, is
seemingly handled by the driver, but then no more IRQs ever arrive and
the device seems to have frozen. Which class of error is more likely
depends on if the device uses edge-triggered or level-triggered
interrupts. A way to check if this has happened is to add a timeout
when waiting for interrupts.

## Interrupts on Linux AMD64

The signal handlers are in `(loko arch amd64 linux-start)`.

Signals under Linux are similar to interrupts. Just like with
interrupts, some signals are traps (BUS, SEGV, FPE, ILL, etc). Linux
places the processor's trap number in the sigcontext of the signal
handler, which makes it easy to handle them identically to the way
traps are handled on bare hardware.

Other signals are from external sources and the program is innocent,
so the current stack frame must be preserved. On bare hardware this is
accomplished by the IST and the equivalent mechanism on Linux is
called sigaltstack(2).

The normal userspace ABI, documented in "System V Application Binary
Interface AMD64 Architecture Processor Supplement", is not followed.
Normally interrupts would be delivered on the same stack as the
program is currently using, but that would mess up Scheme code due to
the way Loko manages stack frames. The "red zone" concept does not
exist in Loko.
