;; -*- mode: scheme; coding: utf-8 -*-
;; SPDX-License-Identifier: AGPL-3.0-or-later
;; Loko Scheme - an R6RS Scheme compiler
;; Copyright © 2019-2020 Göran Weinholt

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU Affero General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU Affero General Public License for more details.

;; You should have received a copy of the GNU Affero General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
#!r6rs

;;;

(library (loko runtime scheduler)
  (export
    ;; Memory handling
    dma-allocate
    dma-free

    ;; IRQ handling
    enable-irq
    disable-irq
    acknowledge-irq
    wait-irq-operation

    ;; Misc calls to the scheduler
    get-environment
    get-pid
    get-boot-loader
    get-boot-modules
    get-command-line
    scheduler-wait
    new-process
    process-exit

    ;; Process identifiers
    pid? pid-value

    ;; XXX: Stands to be cleaned up
    pc-current-ticks
    )
  (import
    (rnrs (6))
    (loko match)
    (loko system $primitives)
    (loko runtime fibers)
    (only (loko system $x86) $enable-interrupts $disable-interrupts)
    (except (loko system $host) dma-allocate dma-free
            enable-irq disable-irq acknowledge-irq wait-irq-operation)
    (only (loko runtime context) CPU-VECTOR:SCHEDULER-RUNNING?
          CPU-VECTOR:SCHEDULER-SP))

(define ($process-yield msg)
  (let ((sched-sp ($processor-data-ref CPU-VECTOR:SCHEDULER-SP)))
    ;; sched-sp is 0 if the scheduler is running
    ;; (display "Yielding back to SCHED-SP=")
    ;; (display (number->string sched-sp 16))
    ;; (newline)
    (when (eqv? sched-sp 0)
      (error '$process-yield "The scheduler tried to yield"))
    ;; IRQs should be disabled when the scheduler is running
    ($disable-interrupts)
    ($processor-data-set! CPU-VECTOR:SCHEDULER-RUNNING? #t) ;currently yielding
    (let ((msg ($switch-stack sched-sp msg)))
      ($processor-data-set! CPU-VECTOR:SCHEDULER-RUNNING? #f) ;no longer yielding
      ($enable-interrupts)    ;FIXME: should not be used under Linux
      ;; (display "Secret code from scheduler: ")
      ;; (write msg)
      ;; (newline)
      msg)))

(define-record-type pid
  (sealed #t) (opaque #f)
  (fields value))

(define (get-command-line)
  (let ((cmdline ($process-yield '(command-line))))
    (assert (list? cmdline))
    (map string-copy cmdline)))

(define (get-environment)
  (let ((env ($process-yield '(environment))))
    (assert (list? env))
    (map (lambda (var)
           (cons (string-copy (car var))
                 (string-copy (cdr var))))
         env)))

(define (get-boot-modules)
  (let ((modules ($process-yield '(boot-modules))))
    (assert (list? modules))
    (map (match-lambda
          [((? string? fn) _args (? fixnum? start) (? fixnum? len))
           (list (string-copy fn) '() start len)])
         modules)))

(define (get-pid)
  (let ((id ($process-yield '(get-pid))))
    (assert (fixnum? id))
    (make-pid id)))

(define (get-boot-loader)
  ($process-yield '(boot-loader)))

(define (new-process)
  (let ((status ($process-yield (vector 'new-process #f))))
    (assert (eq? status 'ok))
    (make-pid (vector-ref status 1))))

(define (process-exit status)
  ($process-yield `(exit ,(pid-value (get-pid))
                         ,status)))

(define (pc-current-ticks)
  ($process-yield '(current-ticks)))

(define (dma-allocate size mask)
  (assert (fixnum? size))
  (assert (fixnum? mask))             ;#xfffff000 is a nice mask
  (let* ((v `#(allocate ,size ,mask #f))
         (s ($process-yield v)))
    (unless (eq? s 'ok)
      (error 'dma-allocate "Memory allocation failed" size mask))
    (let ((cpu-addr (vector-ref v 3)))
      cpu-addr)))

(define (dma-free addr)
  (assert (fixnum? addr))
  (let* ((v `#(free ,addr))
         (s ($process-yield v)))
    (unless (eq? s 'ok)
      (error 'dma-free "Memory release failed" addr))))

(define (scheduler-wait ns-timeout)
  ;; TODO: this message should take a number back from the scheduler
  ;; that indicates for how long it slept. This is so that if the
  ;; process is awoken by an uninteresting message it can go back to
  ;; sleeping with the original timeout without asking the scheduler
  ;; for the current time.
  (when ns-timeout
    (assert (fxpositive? ns-timeout)))
  (let ((vec (vector 'wait ns-timeout #f)))
    (handle-scheduler-wait-reply vec ($process-yield vec)
                                 'scheduler-wait)))

;;; IRQ support

(define *interrupt-cvars* (make-vector 256 #f))

(define (handle-scheduler-wait-reply vec reply who)
  ;; The scheduler will update vec.
  (case reply
    ((timeout)
     #f)
    ((message)
     (let ((msg (vector-ref vec 2)))
       ;; This is currently just IRQ vector numbers, but can be used to
       ;; implement message passing between processes.
       (cond ((and (fixnum? msg) (vector-ref *interrupt-cvars* msg))
              => signal-cvar!))))
    (else
     (error who "Unknown reply from scheduler" reply))))

(define (enable-irq irq)
  (assert (fx<=? 0 irq 15))
  (unless (vector-ref *interrupt-cvars* irq)
    (vector-set! *interrupt-cvars* irq (make-cvar)))
  ($process-yield `#(enable-irq ,irq)))

(define (disable-irq irq)
  (assert (fx<=? 0 irq 15))
  ($process-yield `#(disable-irq ,irq)))

(define (acknowledge-irq irq)
  (define timeout 0)
  (assert (fx<=? 0 irq 15))
  (when timeout
    (assert (not (fxnegative? timeout))))
  (let ((vec (vector 'wait timeout #f)))
    (vector-set! *interrupt-cvars* irq (make-cvar))
    (handle-scheduler-wait-reply vec ($process-yield `#(acknowledge-irq ,irq ,vec))
                                 'acknowledge-irq)))

(define (wait-irq-operation irq)
  (wait-operation (vector-ref *interrupt-cvars* irq))))
